<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Category;
use App\Models\Size;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    public function index()
    {
        $products = Product::with(['category', 'sizes'])->latest()->get();
        return view('admin.products.index', compact('products'));
    }

    public function create()
    {
        $categories = Category::where('is_active', true)->get();
        $sizes = Size::where('is_active', true)->orderBy('order')->get();
        return view('admin.products.create', compact('categories', 'sizes'));
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'category_id' => 'required|exists:categories,id',
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'price' => 'required|numeric|min:0',
                'stock' => 'required|integer|min:0',
                'image' => 'required|file|image|mimes:jpeg,png,jpg,webp|max:5120',
                'points_earned' => 'nullable|integer|min:0',
                'is_active' => 'boolean',
                'is_custom' => 'boolean',
                'sizes' => 'nullable|array',
                'sizes.*' => 'exists:sizes,id',
                'size_stocks' => 'nullable|array',
                'size_prices' => 'nullable|array',
            ], [
                'image.required' => 'يجب رفع صورة للمنتج',
                'image.image' => 'الملف المرفوع يجب أن يكون صورة',
                'image.mimes' => 'نوع الصورة يجب أن يكون: jpeg, png, jpg, webp',
                'image.max' => 'حجم الصورة يجب أن يكون أقل من 5 ميجابايت',
            ]);

            $data = [
                'category_id' => $request->category_id,
                'name' => $request->name,
                'description' => $request->description,
                'price' => $request->price,
                'stock' => $request->stock,
                'points_earned' => $request->points_earned ?? 0,
                'is_active' => $request->has('is_active'),
                'is_custom' => $request->has('is_custom'),
            ];

            if ($request->hasFile('image')) {
                $data['image'] = $request->file('image')->store('products', 'public');
            }

            $product = Product::create($data);

            // إضافة القياسات المرتبطة بالمنتج
            if ($request->has('sizes') && is_array($request->sizes)) {
                $sizesData = [];
                $totalSizesStock = 0;
                
                foreach ($request->sizes as $sizeId) {
                    $stock = $request->size_stocks[$sizeId] ?? 0;
                    $price = $request->size_prices[$sizeId] ?? null;
                    
                    $sizesData[$sizeId] = [
                        'stock' => $stock,
                        'price' => $price,
                    ];
                    
                    $totalSizesStock += $stock;
                }
                
                $product->sizes()->sync($sizesData);
                
                // تحديث المخزون الأساسي للمنتج ليكون مجموع القياسات
                $product->stock = $totalSizesStock;
                $product->save();
            }

            return redirect()->route('admin.products.index')
                ->with('success', 'تم إضافة المنتج بنجاح');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'حدث خطأ أثناء إضافة المنتج: ' . $e->getMessage()]);
        }
    }

    public function edit($id)
    {
        $product = Product::with('sizes')->findOrFail($id);
        $categories = Category::where('is_active', true)->get();
        $sizes = Size::where('is_active', true)->orderBy('order')->get();
        return view('admin.products.edit', compact('product', 'categories', 'sizes'));
    }

    public function update(Request $request, $id)
    {
        $product = Product::findOrFail($id);
        
        $request->validate([
            'category_id' => 'required|exists:categories,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'stock' => 'required|integer|min:0',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'points_earned' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
            'is_custom' => 'boolean',
            'sizes' => 'nullable|array',
            'sizes.*' => 'exists:sizes,id',
            'size_stocks' => 'nullable|array',
            'size_prices' => 'nullable|array',
        ]);

        $product->category_id = $request->category_id;
        $product->name = $request->name;
        $product->description = $request->description;
        $product->price = $request->price;
        $product->stock = $request->stock;
        $product->points_earned = $request->points_earned ?? 0;
        $product->is_active = $request->has('is_active');
        $product->is_custom = $request->has('is_custom');

        if ($request->hasFile('image')) {
            if ($product->image) {
                Storage::disk('public')->delete($product->image);
            }
            $product->image = $request->file('image')->store('products', 'public');
        }

        // تحديث القياسات المرتبطة بالمنتج
        if ($request->has('sizes') && is_array($request->sizes)) {
            $sizesData = [];
            $totalSizesStock = 0;
            
            foreach ($request->sizes as $sizeId) {
                $stock = $request->size_stocks[$sizeId] ?? 0;
                $price = $request->size_prices[$sizeId] ?? null;
                
                $sizesData[$sizeId] = [
                    'stock' => $stock,
                    'price' => $price,
                ];
                
                $totalSizesStock += $stock;
            }
            
            $product->sizes()->sync($sizesData);
            
            // تحديث المخزون الأساسي للمنتج ليكون مجموع القياسات
            $product->stock = $totalSizesStock;
        } else {
            $product->sizes()->detach();
            // إذا لم يكن هناك قياسات، نستخدم المخزون المدخل يدوياً
        }

        $product->save();
        
        // التأكد من تحديث المخزون من القياسات بعد الحفظ
        if ($product->sizes()->count() > 0) {
            $product->updateStockFromSizes();
        }

        return redirect()->route('admin.products.index')
            ->with('success', 'تم تحديث المنتج بنجاح');
    }

    public function destroy($id)
    {
        $product = Product::findOrFail($id);
        
        if ($product->image) {
            Storage::disk('public')->delete($product->image);
        }
        
        $product->delete();

        return redirect()->route('admin.products.index')
            ->with('success', 'تم حذف المنتج بنجاح');
    }
}

