<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Slider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SliderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $sliders = Slider::orderBy('order')->latest()->get();
        return view('admin.sliders.index', compact('sliders'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.sliders.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Check if adding an active slider would exceed the limit
        if ($request->has('is_active')) {
            $activeCount = Slider::where('is_active', true)->count();
            if ($activeCount >= 3) {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['is_active' => 'لا يمكن إضافة أكثر من 3 صور نشطة. يرجى تعطيل إحدى الصور الموجودة أولاً.']);
            }
        }
        
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'required|image|mimes:jpeg,png,jpg,webp|max:2048',
            'link' => 'nullable|url',
            'order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        $imagePath = $request->file('image')->store('sliders', 'public');

        Slider::create([
            'title' => $request->title,
            'description' => $request->description,
            'image' => $imagePath,
            'link' => $request->link,
            'order' => $request->order ?? 0,
            'is_active' => $request->has('is_active'),
        ]);

        return redirect()->route('admin.sliders.index')
            ->with('success', 'تم إضافة السلايدر بنجاح');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $slider = Slider::findOrFail($id);
        return view('admin.sliders.edit', compact('slider'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $slider = Slider::findOrFail($id);
        
        // Check if activating this slider would exceed the limit
        if ($request->has('is_active') && !$slider->is_active) {
            $activeCount = Slider::where('is_active', true)->where('id', '!=', $id)->count();
            if ($activeCount >= 3) {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['is_active' => 'لا يمكن تفعيل أكثر من 3 صور. يرجى تعطيل إحدى الصور الموجودة أولاً.']);
            }
        }
        
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'link' => 'nullable|url',
            'order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        if ($request->hasFile('image')) {
            // Delete old image
            if ($slider->image) {
                Storage::disk('public')->delete($slider->image);
            }
            $imagePath = $request->file('image')->store('sliders', 'public');
            $slider->image = $imagePath;
        }

        $slider->title = $request->title;
        $slider->description = $request->description;
        $slider->link = $request->link;
        $slider->order = $request->order ?? 0;
        $slider->is_active = $request->has('is_active');
        $slider->save();

        return redirect()->route('admin.sliders.index')
            ->with('success', 'تم تحديث السلايدر بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $slider = Slider::findOrFail($id);
        
        // Delete image
        if ($slider->image) {
            Storage::disk('public')->delete($slider->image);
        }
        
        $slider->delete();

        return redirect()->route('admin.sliders.index')
            ->with('success', 'تم حذف السلايدر بنجاح');
    }
}
