<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Cart;
use App\Models\UserPoint;
use App\Mail\OrderConfirmation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;

class OrderController extends Controller
{
    public function index()
    {
        // Prevent admins from viewing customer orders
        if (Auth::check() && Auth::user()->isAdmin()) {
            return redirect()->route('admin.sliders.index')
                ->with('error', app()->getLocale() === 'ar' 
                    ? 'حساب المدير مخصص لإدارة المتجر فقط.' 
                    : 'Admin account is for store management only.');
        }

        $orders = Order::where('user_id', Auth::id())
            ->with('items')
            ->latest()
            ->get();
        
        return view('orders.index', compact('orders'));
    }

    public function create()
    {
        // Require authentication for checkout
        if (!Auth::check()) {
            return redirect()->route('login', ['redirect' => route('orders.create'), 'lang' => 'ar'])->with('error', __('messages.login_required') ?? 'Please login to checkout');
        }

        // Prevent admins from creating orders
        if (Auth::user()->isAdmin()) {
            return redirect()->route('admin.sliders.index')
                ->with('error', app()->getLocale() === 'ar' 
                    ? 'حساب المدير مخصص لإدارة المتجر فقط. لا يمكنك إضافة طلبات.' 
                    : 'Admin account is for store management only. You cannot place orders.');
        }

        $cartItems = Cart::where('user_id', Auth::id())
            ->with('product')
            ->get();
        
        if ($cartItems->isEmpty()) {
            return redirect()->route('cart.index')->with('error', __('messages.cart_empty'));
        }
        
        $total = $cartItems->sum(function($item) {
            return $item->quantity * $item->product->price;
        });
        
        return view('orders.create', compact('cartItems', 'total'));
    }

    public function store(Request $request)
    {
        // Prevent admins from creating orders
        if (Auth::check() && Auth::user()->isAdmin()) {
            return redirect()->route('admin.sliders.index')
                ->with('error', app()->getLocale() === 'ar' 
                    ? 'حساب المدير مخصص لإدارة المتجر فقط. لا يمكنك إضافة طلبات.' 
                    : 'Admin account is for store management only. You cannot place orders.');
        }

        $request->validate([
            'shipping_name' => 'required|string|max:255',
            'shipping_phone' => 'required|string|max:20',
            'shipping_address' => 'required|string',
            'shipping_city' => 'required|string|max:255',
            'notes' => 'nullable|string',
        ]);

        $cartItems = Cart::where('user_id', Auth::id())
            ->with(['product', 'product.sizes', 'size'])
            ->get();
        
        if ($cartItems->isEmpty()) {
            return back()->with('error', __('messages.cart_empty'));
        }

        DB::beginTransaction();
        try {
            $totalAmount = 0;
            $totalPoints = 0;

            // التحقق من المخزون قبل إنشاء الطلب
            foreach ($cartItems as $item) {
                $product = $item->product;
                $availableStock = $product->stock;
                
                // إذا كان المنتج له قياسات وتم اختيار قياس
                if ($item->size_id && $product->sizes->count() > 0) {
                    $selectedSize = $product->sizes->where('id', $item->size_id)->first();
                    if ($selectedSize) {
                        $availableStock = $selectedSize->pivot->stock ?? 0;
                    }
                }
                
                if ($availableStock < $item->quantity) {
                    throw new \Exception(__('messages.out_of_stock') . ': ' . $product->name . ($item->size ? ' (القياس: ' . $item->size->name . ')' : ''));
                }
                
                // حساب السعر مع رسوم التفصيل إذا كان التصنيف 8
                $itemPrice = $product->price;
                if ($product->category_id == 8 && $item->custom_measurements) {
                    $itemPrice += 5; // إضافة 5 ريال رسوم تفصيل
                }
                
                $totalAmount += $item->quantity * $itemPrice;
                $totalPoints += $item->quantity * $product->points_earned;
            }

            $order = Order::create([
                'user_id' => Auth::id(),
                'order_number' => Order::generateOrderNumber(),
                'status' => 'pending',
                'total_amount' => $totalAmount,
                'points_earned' => $totalPoints,
                'shipping_name' => $request->shipping_name,
                'shipping_phone' => $request->shipping_phone,
                'shipping_address' => $request->shipping_address,
                'shipping_city' => $request->shipping_city,
                'notes' => $request->notes,
            ]);

            foreach ($cartItems as $item) {
                $product = $item->product;
                
                // تحضير بيانات القياسات المخصصة
                $customMeasurements = null;
                if ($item->custom_measurements) {
                    if (is_array($item->custom_measurements)) {
                        $customMeasurements = $item->custom_measurements;
                    } elseif (is_string($item->custom_measurements)) {
                        $decoded = json_decode($item->custom_measurements, true);
                        if (is_array($decoded)) {
                            $customMeasurements = $decoded;
                        }
                    }
                }
                
                // حساب السعر مع رسوم التفصيل إذا كان التصنيف 8
                $itemPrice = $product->price;
                if ($product->category_id == 8 && $customMeasurements) {
                    $itemPrice += 5; // إضافة 5 ريال رسوم تفصيل
                }
                
                OrderItem::create([
                    'order_id' => $order->id,
                    'product_id' => $item->product_id,
                    'size_id' => $item->size_id,
                    'product_name' => $product->name,
                    'price' => $itemPrice,
                    'quantity' => $item->quantity,
                    'subtotal' => $item->quantity * $itemPrice,
                    'custom_measurements' => $customMeasurements,
                ]);

                // خصم المخزون من القياس المحدد إذا كان موجوداً
                if ($item->size_id && $product->sizes->count() > 0) {
                    $selectedSize = $product->sizes->where('id', $item->size_id)->first();
                    if ($selectedSize) {
                        // خصم من مخزون القياس
                        $currentSizeStock = $selectedSize->pivot->stock ?? 0;
                        $newSizeStock = max(0, $currentSizeStock - $item->quantity);
                        
                        // تحديث مخزون القياس في pivot table
                        $product->sizes()->updateExistingPivot($item->size_id, [
                            'stock' => $newSizeStock
                        ]);
                        
                        // تحديث المخزون الأساسي للمنتج (مجموع القياسات)
                        $product->updateStockFromSizes();
                    }
                } else {
                    // إذا لم يكن هناك قياس، نخصم من المخزون الأساسي
                    $product->decrement('stock', $item->quantity);
                }
            }

            if ($totalPoints > 0) {
                Auth::user()->increment('points', $totalPoints);
                
                UserPoint::create([
                    'user_id' => Auth::id(),
                    'order_id' => $order->id,
                    'points' => $totalPoints,
                    'type' => 'earned',
                    'description' => 'Points earned from order #' . $order->order_number,
                ]);
            }

            Cart::where('user_id', Auth::id())->delete();

            DB::commit();

            // Send order confirmation email
            try {
                Mail::to(Auth::user()->email)->send(new OrderConfirmation($order));
            } catch (\Exception $e) {
                // Log error but don't fail the order
                \Log::error('Failed to send order confirmation email: ' . $e->getMessage());
            }

            return redirect()->route('orders.show', $order->id)
                ->with('success', __('messages.order_placed_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', $e->getMessage());
        }
    }

    public function show($id)
    {
        // Prevent admins from viewing customer orders
        if (Auth::check() && Auth::user()->isAdmin()) {
            return redirect()->route('admin.sliders.index')
                ->with('error', app()->getLocale() === 'ar' 
                    ? 'حساب المدير مخصص لإدارة المتجر فقط.' 
                    : 'Admin account is for store management only.');
        }

        $order = Order::where('user_id', Auth::id())
            ->with(['items.product', 'items.size'])
            ->findOrFail($id);
        
        return view('orders.show', compact('order'));
    }

    public function track(Request $request)
    {
        $order = null;
        
        if ($request->has('order_number')) {
            $order = Order::where('order_number', $request->order_number)
                ->with('items.product')
                ->first();
        }
        
        return view('orders.track', compact('order'));
    }
}
