<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Storage;

class Category extends Model
{
    protected $fillable = [
        'name',
        'description',
        'image',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    public function products(): HasMany
    {
        return $this->hasMany(Product::class);
    }

    /**
     * Get the normalized image URL
     * Handles both full URLs and relative paths
     */
    public function getImageUrlAttribute(): ?string
    {
        if (!$this->image) {
            return null;
        }

        // If it's already a full URL, extract the path
        if (str_starts_with($this->image, 'http')) {
            // Extract the path from the URL
            $parsedUrl = parse_url($this->image);
            if (isset($parsedUrl['path'])) {
                // Remove leading slash and storage/ prefix if present
                $path = ltrim($parsedUrl['path'], '/');
                if (str_starts_with($path, 'storage/')) {
                    $path = substr($path, 8); // Remove 'storage/' prefix
                }
                // Use Storage::url() which handles the URL generation correctly
                return Storage::disk('public')->url($path);
            }
            // If we can't parse it, return as is (might be external URL)
            return $this->image;
        }

        // It's a relative path, use Storage::url() which handles it correctly
        $path = ltrim($this->image, '/');
        if (str_starts_with($path, 'storage/')) {
            $path = substr($path, 8); // Remove 'storage/' prefix
        }
        
        return Storage::disk('public')->url($path);
    }

}
